<?php
declare(strict_types=1);

// ------------------------------------------------------------
// No-Composer autoloader (cPanel-friendly)
// ------------------------------------------------------------
spl_autoload_register(function(string $class): void {
    $prefix = 'App\\';
    if (strncmp($class, $prefix, strlen($prefix)) !== 0) {
        return;
    }
    $rel = substr($class, strlen($prefix));
    $file = dirname(__DIR__) . '/app/' . str_replace('\\', '/', $rel) . '.php';
    if (is_file($file)) {
        require $file;
    }
});

$root = dirname(__DIR__);
$configFile = $root . '/config.php';
if (!file_exists($configFile)) {
    header('Location: ../setup.php');
    exit;
}
$config = require $configFile;

date_default_timezone_set($config['app']['timezone'] ?? 'Asia/Riyadh');

// start session for flash
if (session_status() !== PHP_SESSION_ACTIVE) session_start();

use App\Storage\Db;
use App\Utils\Logger;
use App\Http\HttpClient;
use App\Storage\ImageDownloader;
use App\Utils\Scraper;
use App\Export\ExcelExporter;

function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function redirect(string $to){ header("Location: $to"); exit; }

$db = new \App\Storage\Db($config['db']);
$logger = new \App\Utils\Logger($db);
$http = new \App\Http\HttpClient($config['app']['user_agent'] ?? 'Mozilla/5.0 (compatible; SallaMigrator/1.0)');
$img = new \App\Storage\ImageDownloader($http, $config['paths']['images']);
$scraper = new \App\Utils\Scraper($http, $db, $logger, $img);

$route = $_GET['r'] ?? 'dashboard';

// Handle POST actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        if (($route === 'scrape') && isset($_POST['category_url']) && $_POST['category_url'] !== '') {
            $count = $scraper->scrapeCategory(trim($_POST['category_url']));
            $_SESSION['flash'] = "تم جلب $count منتج من صفحة التصنيف (حفظ أولي بدون تفاصيل).";
            redirect('?r=products');
        }

        if (($route === 'scrape') && isset($_POST['product_url']) && $_POST['product_url'] !== '') {
            $pid = $scraper->scrapeProduct(trim($_POST['product_url']), true);
            $_SESSION['flash'] = "تم سحب المنتج وتحديثه. ID داخلي: $pid";
            redirect('?r=product&id=' . $pid);
        }

        if (($route === 'scrape') && isset($_FILES['html_file']) && $_FILES['html_file']['error'] === UPLOAD_ERR_OK) {
            $tmp = $_FILES['html_file']['tmp_name'];
            $name = basename($_FILES['html_file']['name']);
            $dest = rtrim($config['paths']['uploads'], '/') . '/' . preg_replace('/[^a-zA-Z0-9\-_\.]+/u','_', $name);
            move_uploaded_file($tmp, $dest);
            $pid = $scraper->scrapeProductHtmlFile($dest, false);
            $_SESSION['flash'] = "تم تحليل ملف HTML وحفظ المنتج. ID داخلي: $pid";
            redirect('?r=product&id=' . $pid);
        }

        if ($route === 'export') {
            $exporter = new \App\Export\ExcelExporter(
                $db,
                $root . '/Salla Products Template.xlsx',
                $config['paths']['exports'],
                (bool)($config['app']['tax_enabled'] ?? true),
                $_POST['images_mode'] ?? 'first'
            );
            $path = $exporter->export();
            $_SESSION['flash'] = "تم إنشاء ملف التصدير: " . basename($path);
            redirect('?r=export&done=1&file=' . urlencode(basename($path)));
        }
    } catch (Throwable $e) {
        $_SESSION['flash_error'] = $e->getMessage();
        redirect('?r=' . urlencode($route));
    }
}


$flash = $_SESSION['flash'] ?? null;
$flashErr = $_SESSION['flash_error'] ?? null;
unset($_SESSION['flash'], $_SESSION['flash_error']);

$stats = $db->stats();

function layout(string $title, callable $body){
    global $flash, $flashErr, $stats;
    ?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title><?=h($title)?></title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.datatables.net/1.13.6/css/dataTables.bootstrap5.min.css" rel="stylesheet">
  <style>
    body { background:#f6f7fb; }
    .card { border:0; border-radius:16px; }
    .navbar { border-radius:16px; }
    .badge-soft { background:#eef2ff; color:#334155; }
    .table td, .table th { vertical-align: middle; }
    .product-img { width:56px; height:56px; object-fit:cover; border-radius:12px; background:#fff; }
  </style>
</head>
<body>
<div class="container py-4">
  <nav class="navbar navbar-expand-lg bg-white shadow-sm px-3 mb-4">
    <a class="navbar-brand fw-bold" href="?r=dashboard">Salla Migrator</a>
    <div class="ms-auto d-flex gap-2">
      <a class="btn btn-outline-primary btn-sm" href="?r=scrape">Scrape</a>
      <a class="btn btn-outline-primary btn-sm" href="?r=products">Products</a>
      <a class="btn btn-outline-primary btn-sm" href="?r=export">Export</a>
      <a class="btn btn-outline-secondary btn-sm" href="../setup.php">Setup</a>
    </div>
  </nav>

  <?php if ($flash): ?><div class="alert alert-success"><?=h($flash)?></div><?php endif; ?>
  <?php if ($flashErr): ?><div class="alert alert-danger"><?=h($flashErr)?></div><?php endif; ?>

  <?php $body(); ?>

  <div class="text-muted small mt-4">
    إجمالي المنتجات: <b><?= (int)$stats['total'] ?></b> — جديد: <b><?= (int)$stats['new'] ?></b> — مُصدّر: <b><?= (int)$stats['exported'] ?></b> — فشل: <b><?= (int)$stats['failed'] ?></b>
  </div>
</div>

<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.6/js/dataTables.bootstrap5.min.js"></script>
</body>
</html>
<?php
}

if ($route === 'dashboard') {
    layout('Dashboard', function() use ($stats) {
        ?>
        <div class="row g-3">
          <div class="col-md-3"><div class="card shadow-sm"><div class="card-body">
            <div class="text-muted">إجمالي</div><div class="display-6 fw-bold"><?= (int)$stats['total'] ?></div>
          </div></div></div>
          <div class="col-md-3"><div class="card shadow-sm"><div class="card-body">
            <div class="text-muted">جديد</div><div class="display-6 fw-bold"><?= (int)$stats['new'] ?></div>
          </div></div></div>
          <div class="col-md-3"><div class="card shadow-sm"><div class="card-body">
            <div class="text-muted">مُصدّر</div><div class="display-6 fw-bold"><?= (int)$stats['exported'] ?></div>
          </div></div></div>
          <div class="col-md-3"><div class="card shadow-sm"><div class="card-body">
            <div class="text-muted">فشل</div><div class="display-6 fw-bold"><?= (int)$stats['failed'] ?></div>
          </div></div></div>
        </div>

        <div class="card shadow-sm mt-3">
          <div class="card-body">
            <h5 class="mb-2">كيف تبدأ؟</h5>
            <ol class="mb-0">
              <li>اذهب إلى صفحة <b>Scrape</b> وأدخل رابط تصنيف أو رابط منتج.</li>
              <li>راجع المنتجات في <b>Products</b> (لن يحدث تكرار بسبب قيود UNIQUE).</li>
              <li>اذهب إلى <b>Export</b> واستخرج ملف Excel مطابق لقالب سلة.</li>
            </ol>
          </div>
        </div>
        <?php
    });
    exit;
}

if ($route === 'products') {
    $q = $_GET['q'] ?? '';
    $status = $_GET['status'] ?? '';
    $products = $db->getProducts(['q' => $q, 'status' => $status ?: null]);

    layout('Products', function() use ($products, $q, $status) {
        ?>
        <div class="card shadow-sm mb-3">
          <div class="card-body">
            <form class="row g-2" method="get">
              <input type="hidden" name="r" value="products">
              <div class="col-md-6">
                <input class="form-control" name="q" value="<?=h($q)?>" placeholder="بحث بالاسم أو SKU أو الرابط">
              </div>
              <div class="col-md-3">
                <select class="form-select" name="status">
                  <option value="">كل الحالات</option>
                  <option value="new" <?= $status==='new'?'selected':'' ?>>new</option>
                  <option value="exported" <?= $status==='exported'?'selected':'' ?>>exported</option>
                  <option value="failed" <?= $status==='failed'?'selected':'' ?>>failed</option>
                </select>
              </div>
              <div class="col-md-3 d-grid">
                <button class="btn btn-primary">فلترة</button>
              </div>
            </form>
          </div>
        </div>

        <div class="card shadow-sm">
          <div class="card-body">
            <div class="table-responsive">
              <table id="t" class="table table-hover align-middle">
                <thead>
                  <tr>
                    <th>صورة</th>
                    <th>الاسم</th>
                    <th>السعر</th>
                    <th>SKU</th>
                    <th>الحالة</th>
                    <th>تحديث</th>
                  </tr>
                </thead>
                <tbody>
                <?php foreach ($products as $p): ?>
                  <tr>
                    <td>
                      <?php if (!empty($p['main_image_url'])): ?>
                        <img class="product-img" src="<?=h($p['main_image_url'])?>" alt="">
                      <?php else: ?>
                        <div class="product-img d-flex align-items-center justify-content-center text-muted">—</div>
                      <?php endif; ?>
                    </td>
                    <td>
                      <a class="fw-semibold text-decoration-none" href="?r=product&id=<?= (int)$p['id'] ?>"><?=h($p['name'])?></a>
                      <div class="small text-muted text-truncate" style="max-width:520px;"><?=h($p['source_url'])?></div>
                    </td>
                    <td><?=h($p['price'] ?? '')?></td>
                    <td><?=h($p['sku'] ?? '')?></td>
                    <td><span class="badge badge-soft"><?=h($p['status'])?></span></td>
                    <td class="text-end">
                      <a class="btn btn-sm btn-outline-primary" href="?r=scrape&prefill=<?=urlencode($p['source_url'])?>">سحب تفاصيل</a>
                    </td>
                  </tr>
                <?php endforeach; ?>
                </tbody>
              </table>
            </div>
          </div>
        </div>

        <script>
          $(function(){ $('#t').DataTable({pageLength: 25, order:[[5,'desc']]}); });
        </script>
        <?php
    });
    exit;
}

if ($route === 'product') {
    $id = (int)($_GET['id'] ?? 0);
    $p = $db->getProduct($id);

    layout('Product', function() use ($p) {
        if (!$p) { echo '<div class="alert alert-warning">المنتج غير موجود.</div>'; return; }
        ?>
        <div class="card shadow-sm mb-3">
          <div class="card-body">
            <div class="d-flex gap-3 align-items-start">
              <?php if (!empty($p['main_image_url'])): ?>
                <img class="product-img" style="width:88px;height:88px;" src="<?=h($p['main_image_url'])?>">
              <?php endif; ?>
              <div class="flex-grow-1">
                <div class="d-flex justify-content-between">
                  <div>
                    <h4 class="mb-1"><?=h($p['name'])?></h4>
                    <div class="text-muted small"><?=h($p['source_url'])?></div>
                  </div>
                  <div class="text-end">
                    <div class="fs-4 fw-bold"><?=h($p['price'] ?? '')?> <?=h($p['currency'] ?? 'SAR')?></div>
                    <span class="badge badge-soft"><?=h($p['status'])?></span>
                  </div>
                </div>
                <div class="mt-2 small">
                  <b>SKU:</b> <?=h($p['sku'] ?? '')?> &nbsp; | &nbsp;
                  <b>Brand:</b> <?=h($p['brand'] ?? '')?>
                </div>
                <div class="mt-2 small">
                  <b>Categories:</b> <?=h(implode(' > ', $p['categories'] ?? []))?>
                </div>
              </div>
            </div>
          </div>
        </div>

        <div class="row g-3">
          <div class="col-lg-7">
            <div class="card shadow-sm">
              <div class="card-body">
                <h6 class="mb-2">الوصف</h6>
                <div class="border rounded-3 p-3 bg-white" style="max-height:420px; overflow:auto;">
                  <?= $p['description_html'] ?: nl2br(h($p['description_text'] ?? '')) ?>
                </div>
              </div>
            </div>
          </div>
          <div class="col-lg-5">
            <div class="card shadow-sm">
              <div class="card-body">
                <h6 class="mb-2">الصور (<?= (int)count($p['images'] ?? []) ?>)</h6>
                <div class="d-flex flex-wrap gap-2">
                  <?php foreach (($p['images'] ?? []) as $im): ?>
                    <a href="<?=h($im['image_url'])?>" target="_blank">
                      <img class="product-img" src="<?=h($im['image_url'])?>" alt="">
                    </a>
                  <?php endforeach; ?>
                </div>
              </div>
            </div>
          </div>
        </div>
        <?php
    });
    exit;
}

if ($route === 'scrape') {
    $prefill = $_GET['prefill'] ?? '';
    layout('Scrape', function() use ($prefill) {
        ?>
        <div class="row g-3">
          <div class="col-lg-6">
            <div class="card shadow-sm">
              <div class="card-body">
                <h5 class="mb-3">سحب من رابط تصنيف</h5>
                <form method="post">
                  <input class="form-control mb-2" name="category_url" placeholder="ضع رابط التصنيف" value="">
                  <button class="btn btn-primary">سحب روابط المنتجات (حفظ أولي)</button>
                  <div class="text-muted small mt-2">هذه الخطوة تجلب قائمة المنتجات من التصنيف وتحفظها دون تفاصيل كاملة.</div>
                </form>
              </div>
            </div>
          </div>

          <div class="col-lg-6">
            <div class="card shadow-sm">
              <div class="card-body">
                <h5 class="mb-3">سحب منتج كامل من رابط منتج</h5>
                <form method="post">
                  <input class="form-control mb-2" name="product_url" placeholder="ضع رابط المنتج" value="<?=h($prefill)?>">
                  <button class="btn btn-primary">سحب تفاصيل المنتج + الصور</button>
                </form>
              </div>
            </div>
          </div>

          <div class="col-lg-12">
            <div class="card shadow-sm">
              <div class="card-body">
                <h5 class="mb-3">رفع ملف HTML لمنتج (للاختبار)</h5>
                <form method="post" enctype="multipart/form-data">
                  <input class="form-control mb-2" type="file" name="html_file" accept=".html,.htm" required>
                  <button class="btn btn-outline-primary">رفع وتحليل</button>
                  <div class="text-muted small mt-2">رفع ملفات HTML مثل التي أرسلتها لاختبار الاستخراج بدون الاتصال بالموقع.</div>
                </form>
              </div>
            </div>
          </div>
        </div>
        <?php
    });
    exit;
}

if ($route === 'export') {
    $done = isset($_GET['done']);
    $file = $_GET['file'] ?? '';
    layout('Export', function() use ($done, $file) {
        ?>
        <div class="card shadow-sm">
          <div class="card-body">
            <h5 class="mb-3">تصدير Excel مطابق لقالب سلة</h5>

            <form method="post" class="row g-2">
              <div class="col-md-4">
                <label class="form-label">وضع الصور داخل Excel</label>
                <select class="form-select" name="images_mode">
                  <option value="first">صورة واحدة (الأولى)</option>
                  <option value="all">كل الصور (سطر جديد)</option>
                </select>
              </div>
              <div class="col-md-8 d-flex align-items-end">
                <button class="btn btn-primary">Export Now</button>
              </div>
            </form>

            <?php if ($done && $file): ?>
              <hr>
              <div class="alert alert-success">
                تم إنشاء الملف: <b><?=h($file)?></b><br>
                <a class="btn btn-success btn-sm mt-2" href="../storage/exports/<?=h($file)?>" download>تحميل ملف Excel</a>
              </div>
            <?php endif; ?>

            <div class="text-muted small mt-2">
              بعد تحميل الملف: اذهب إلى لوحة سلة &gt; المنتجات &gt; استيراد، وارفع ملف Excel.
            </div>
          </div>
        </div>
        <?php
    });
    exit;
}

// default
redirect('?r=dashboard');
