<?php
/**
 * Setup script for cPanel.
 * 1) Copy config.example.php => config.php
 * 2) Run database.sql migrations
 *
 * Usage (browser): /setup.php
 */

$root = __DIR__;

if (!file_exists($root . '/config.php')) {
    copy($root . '/config.example.php', $root . '/config.php');
}

$config = require $root . '/config.php';

function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

$error = null;
$ok = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Update config.php
    $config['db']['host'] = trim($_POST['db_host'] ?? 'localhost');
    $config['db']['name'] = trim($_POST['db_name'] ?? '');
    $config['db']['user'] = trim($_POST['db_user'] ?? '');
    $config['db']['pass'] = (string)($_POST['db_pass'] ?? '');
    $config['app']['base_url'] = trim($_POST['base_url'] ?? '');
    $config['app']['tax_enabled'] = isset($_POST['tax_enabled']);

    $export = "<?php\nreturn " . var_export($config, true) . ";\n";
    file_put_contents($root . '/config.php', $export);

    try {
        $dsn = "mysql:host={$config['db']['host']};dbname={$config['db']['name']};charset={$config['db']['charset']}";
        $pdo = new PDO($dsn, $config['db']['user'], $config['db']['pass'], [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]);

        $sql = file_get_contents($root . '/database.sql');
        $pdo->exec($sql);
        $ok[] = 'Database schema installed.';

        // Create storage directories
        foreach (['storage','storage/uploads','storage/images','storage/exports'] as $dir) {
            $path = $root . '/' . $dir;
            if (!is_dir($path)) mkdir($path, 0775, true);
        }
        $ok[] = 'Storage directories ready.';

        // Write .htaccess for public/ if missing
        $ht = $root . '/public/.htaccess';
        if (!file_exists($ht)) {
            file_put_contents($ht, "Options -Indexes\nRewriteEngine On\nRewriteCond %{REQUEST_FILENAME} !-f\nRewriteCond %{REQUEST_FILENAME} !-d\nRewriteRule ^ index.php [QSA,L]\n");
            $ok[] = 'public/.htaccess created.';
        }

    } catch (Throwable $e) {
        $error = $e->getMessage();
    }
}

?><!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Salla Migrator Setup</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light">
<div class="container py-5" style="max-width: 900px;">
  <div class="card shadow-sm">
    <div class="card-body">
      <h3 class="mb-3">إعداد Salla Migrator</h3>
      <p class="text-muted">املأ بيانات قاعدة البيانات (من cPanel) ثم اضغط تثبيت.</p>

      <?php if ($error): ?>
        <div class="alert alert-danger">خطأ: <?=h($error)?></div>
      <?php endif; ?>
      <?php foreach ($ok as $m): ?>
        <div class="alert alert-success"><?=h($m)?></div>
      <?php endforeach; ?>

      <form method="post" class="row g-3">
        <div class="col-md-6">
          <label class="form-label">DB Host</label>
          <input class="form-control" name="db_host" value="<?=h($config['db']['host'] ?? 'localhost')?>">
        </div>
        <div class="col-md-6">
          <label class="form-label">DB Name</label>
          <input class="form-control" name="db_name" value="<?=h($config['db']['name'] ?? '')?>" required>
        </div>
        <div class="col-md-6">
          <label class="form-label">DB User</label>
          <input class="form-control" name="db_user" value="<?=h($config['db']['user'] ?? '')?>" required>
        </div>
        <div class="col-md-6">
          <label class="form-label">DB Pass</label>
          <input class="form-control" name="db_pass" type="password" value="<?=h($config['db']['pass'] ?? '')?>">
        </div>
        <div class="col-12">
          <label class="form-label">Base URL (اختياري)</label>
          <input class="form-control" name="base_url" value="<?=h($config['app']['base_url'] ?? '')?>" placeholder="https://yourdomain.com/salla_migrator/public">
        </div>
        <div class="col-12">
          <div class="form-check">
            <input class="form-check-input" type="checkbox" name="tax_enabled" id="tax" <?=!empty($config['app']['tax_enabled'])?'checked':''?>>
            <label class="form-check-label" for="tax">متجري يستخدم الضريبة</label>
          </div>
        </div>
        <div class="col-12">
          <button class="btn btn-primary">تثبيت</button>
          <a class="btn btn-outline-secondary" href="public/">فتح لوحة التحكم</a>
        </div>
      </form>

      <hr>
      <div class="small text-muted">
        بعد التثبيت: احذف <code>setup.php</code> لأسباب أمنية.
      </div>
    </div>
  </div>
</div>
</body>
</html>
