<?php
namespace App\Export;

use App\Storage\Db;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Spreadsheet;

class ExcelExporter
{
    public function __construct(
        private Db $db,
        private string $templatePath,
        private string $exportsDir,
        private bool $taxEnabled = true,
        private string $imagesMode = 'first' // first|all
    ) {}

    public function export(): string
    {
        if (!file_exists($this->templatePath)) {
            throw new \RuntimeException('Template not found: ' . $this->templatePath);
        }

        $spreadsheet = IOFactory::load($this->templatePath);
        $sheet = $spreadsheet->getSheetByName('Salla Products Template Sheet') ?? $spreadsheet->getActiveSheet();

        // Headers are on row 2
        $headerRow = 2;
        $colMap = $this->buildColumnMap($sheet, $headerRow);

        // Clear old data rows (from row 3 to last)
        $startRow = 3;
        $maxRow = $sheet->getHighestRow();
        if ($maxRow >= $startRow) {
            $sheet->removeRow($startRow, $maxRow - $startRow + 1);
        }

        $products = $this->db->getProducts(['status' => null]);

        $row = $startRow;
        $exportedIds = [];

        foreach ($products as $p) {
            $pid = (int)$p['id'];
            $details = $this->db->getProduct($pid);
            if (!$details) continue;

            $categoryPath = '';
            if (!empty($details['categories']) && is_array($details['categories'])) {
                // Remove 'Home' if present, keep readable path
                $cats = array_values(array_filter($details['categories'], fn($c) => $c && $c !== 'Home' && $c !== 'الرئيسية'));
                $categoryPath = implode(' > ', $cats);
            }

            $imageUrls = array_map(fn($im) => $im['image_url'], $details['images'] ?? []);
            if (empty($imageUrls) && !empty($details['main_image_url'])) $imageUrls = [$details['main_image_url']];

            $imageCell = '';
            if ($this->imagesMode === 'all') {
                // Some importers accept new lines
                $imageCell = implode("\n", array_slice($imageUrls, 0, 10));
            } else {
                $imageCell = $imageUrls[0] ?? '';
            }

            $this->setCell($sheet, $colMap, 'النوع ', $row, 'منتج');
            $this->setCell($sheet, $colMap, 'أسم المنتج', $row, $details['name'] ?? '');
            $this->setCell($sheet, $colMap, 'تصنيف المنتج', $row, $categoryPath);
            $this->setCell($sheet, $colMap, 'صورة المنتج', $row, $imageCell);
            $this->setCell($sheet, $colMap, 'وصف صورة المنتج', $row, '');
            $this->setCell($sheet, $colMap, 'نوع المنتج', $row, 'منتج جاهز');

            $price = $details['price'] ?? null;
            $this->setCell($sheet, $colMap, 'سعر المنتج', $row, $price);

            // Description: keep html if available
            $desc = $details['description_html'] ?? $details['description_text'] ?? '';
            $this->setCell($sheet, $colMap, 'الوصف', $row, $desc);

            $this->setCell($sheet, $colMap, 'هل يتطلب شحن؟', $row, 'نعم');
            $this->setCell($sheet, $colMap, 'رمز المنتج sku', $row, $details['sku'] ?? '');
            $this->setCell($sheet, $colMap, 'الماركة', $row, $details['brand'] ?? '');

            // Tax
            $this->setCell($sheet, $colMap, 'خاضع للضريبة ؟', $row, $this->taxEnabled ? 'نعم' : 'لا');
            if (!$this->taxEnabled) {
                $this->setCell($sheet, $colMap, 'سبب عدم الخضوع للضريبة', $row, '');
            }

            $row++;
            $exportedIds[] = $pid;
        }

        // Save
        if (!is_dir($this->exportsDir)) mkdir($this->exportsDir, 0775, true);
        $filename = 'salla_export_' . date('Y-m-d_His') . '.xlsx';
        $outPath = rtrim($this->exportsDir, '/') . '/' . $filename;

        IOFactory::createWriter($spreadsheet, 'Xlsx')->save($outPath);

        // mark exported
        $this->db->markExported($exportedIds);

        return $outPath;
    }

    private function buildColumnMap($sheet, int $headerRow): array
    {
        $map = [];
        $highestCol = $sheet->getHighestColumn();
        $highestIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($highestCol);
        for ($col = 1; $col <= $highestIndex; $col++) {
            $val = trim((string)$sheet->getCellByColumnAndRow($col, $headerRow)->getValue());
            if ($val !== '') {
                $map[$val] = $col;
            }
        }
        return $map;
    }

    private function setCell($sheet, array $map, string $header, int $row, $value): void
    {
        if (!isset($map[$header])) return;
        $col = $map[$header];
        $sheet->setCellValueByColumnAndRow($col, $row, $value);
    }
}
