<?php
// هذا الملف مسؤول عن قراءة ملفات Excel الخاصة بالتحويلات البنكية
require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/db.php';

// تحتاج إلى وجود مكتبة PhpSpreadsheet في مجلد vendor
require_once __DIR__ . '/vendor/autoload.php';

use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Shared\Date as ExcelDate;

function process_excel_for_file(int $uploaded_file_id): bool
{
    global $mysqli;

    // جلب بيانات الملف
    $stmt = $mysqli->prepare("SELECT stored_name FROM uploaded_files WHERE id = ? LIMIT 1");
    $stmt->bind_param("i", $uploaded_file_id);
    $stmt->execute();
    $res = $stmt->get_result();
    $fileRow = $res->fetch_assoc();
    $stmt->close();

    if (!$fileRow) {
        log_activity('excel_process_failed', 'لم يتم العثور على ملف للمعالجة ID=' . $uploaded_file_id);
        return false;
    }

    $storedName = $fileRow['stored_name'];
    $filePath = __DIR__ . '/uploads/' . $storedName;

    if (!file_exists($filePath)) {
        log_activity('excel_process_failed', 'الملف غير موجود على السيرفر: ' . $filePath);
        $mysqli->query("UPDATE uploaded_files SET status='failed' WHERE id=" . (int)$uploaded_file_id);
        return false;
    }

    try {
        $spreadsheet = IOFactory::load($filePath);
    } catch (Throwable $e) {
        log_activity('excel_process_failed', 'فشل قراءة ملف Excel: ' . $e->getMessage());
        $mysqli->query("UPDATE uploaded_files SET status='failed' WHERE id=" . (int)$uploaded_file_id);
        return false;
    }

    // نفترض أن:
    // التابة الأولى: التحويلات البنكية (ملخص)
    // التابة الثانية: تقرير تفاصيل التحويلات البنكية
    $summarySheet = $spreadsheet->getSheet(0);
    $detailsSheet = $spreadsheet->getSheetCount() > 1 ? $spreadsheet->getSheet(1) : null;

    // حذف أي بيانات قديمة مرتبطة بنفس الملف
    $mysqli->query("DELETE FROM expenses_headers WHERE uploaded_file_id = " . (int)$uploaded_file_id);
    $mysqli->query("DELETE FROM expenses_details WHERE uploaded_file_id = " . (int)$uploaded_file_id);

    // =========================
    // 1) قراءة التابة الأولى (التحويلات البنكية)
    // =========================
    $fromDate = null;
    $toDate   = null;
    $totalBank = 0.0;
    $totalTms  = 0.0;
    $highestDiff = 0.0;
    $lowestDiff  = 0.0;

    $highestRowSummary = $summarySheet->getHighestDataRow();

    // البحث في الصفوف عن القيم
    for ($row = 1; $row <= $highestRowSummary; $row++) {
        $col1 = trim((string)$summarySheet->getCellByColumnAndRow(1, $row)->getValue());
        $col2 = $summarySheet->getCellByColumnAndRow(2, $row)->getValue();
        $col3 = trim((string)$summarySheet->getCellByColumnAndRow(3, $row)->getValue());
        $col4 = $summarySheet->getCellByColumnAndRow(4, $row)->getValue();

        // سطر الفترة: من | تاريخ من | إلى | تاريخ إلى
        if ($col1 === 'من' && $col3 === 'إلى') {
            $fromDate = parse_excel_date_cell($summarySheet->getCellByColumnAndRow(2, $row));
            $toDate   = parse_excel_date_cell($summarySheet->getCellByColumnAndRow(4, $row));
        }

        // إجمالي تحويل البنك
        if (strpos($col1, 'اجمالى تحويل البنك') !== false) {
            $totalBank = (float)$col2;
        }

        // إجمالي أوامر دفع TMS
        if (strpos($col1, 'اجمالى أوامر دفع TMS') !== false) {
            $totalTms = (float)$col2;
        }

        // تحويل أعلى
        if (strpos($col1, 'تحويل أعلى') !== false) {
            $highestDiff = (float)$col2;
        }

        // تحويل أقل
        if (strpos($col1, 'تحويل أقل') !== false) {
            $lowestDiff = (float)$col2;
        }
    }

    // إدخال صف واحد في جدول الملخص
    $stmtHeader = $mysqli->prepare("
        INSERT INTO expenses_headers
        (uploaded_file_id, from_date, to_date, total_bank, total_tms, highest_diff, lowest_diff)
        VALUES (?, ?, ?, ?, ?, ?, ?)
    ");
    $stmtHeader->bind_param(
        "issdddd",
        $uploaded_file_id,
        $fromDate,
        $toDate,
        $totalBank,
        $totalTms,
        $highestDiff,
        $lowestDiff
    );
    $stmtHeader->execute();
    $stmtHeader->close();

    // =========================
    // 2) قراءة التابة الثانية (تقرير تفاصيل التحويلات البنكية)
    // =========================
    $insertedDetails = 0;
    if ($detailsSheet !== null) {
        $highestRowDetails = $detailsSheet->getHighestDataRow();

        if ($highestRowDetails >= 2) {
            $stmtDetail = $mysqli->prepare("
                INSERT INTO expenses_details
                (uploaded_file_id, row_number, beneficiary_name, owner_number, driver_number, owner_type, iban,
                 tms_amount, bank_amount, diff_amount, transfer_date, beneficiary_note,
                 license_number, purpose, created_at_excel)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");

            for ($row = 2; $row <= $highestRowDetails; $row++) {
                // الأعمدة بالترتيب:
                // 1 #، 2 اسم المستفيد، 3 رقم المالك، 4 رقم السائق، 5 نوع المالك، 6 ايبان،
                // 7 أوامر دفع TMS، 8 تحويل البنك، 9 فرق التحويل، 10 تاريخ التحويل،
                // 11 ملاحظة المستفيد، 12 رقم الترخيص، 13 غرض التحويل، 14 تاريخ الإنشاء.
                $rowNumber = (int)$detailsSheet->getCellByColumnAndRow(1, $row)->getValue();
                $beneficiaryName = trim((string)$detailsSheet->getCellByColumnAndRow(2, $row)->getValue());
                $ownerNumber     = trim((string)$detailsSheet->getCellByColumnAndRow(3, $row)->getValue());
                $driverNumber    = trim((string)$detailsSheet->getCellByColumnAndRow(4, $row)->getValue());
                $ownerType       = trim((string)$detailsSheet->getCellByColumnAndRow(5, $row)->getValue());
                $iban            = trim((string)$detailsSheet->getCellByColumnAndRow(6, $row)->getValue());
                $tmsAmount       = (float)$detailsSheet->getCellByColumnAndRow(7, $row)->getCalculatedValue();
                $bankAmount      = (float)$detailsSheet->getCellByColumnAndRow(8, $row)->getCalculatedValue();
                $diffAmount      = (float)$detailsSheet->getCellByColumnAndRow(9, $row)->getCalculatedValue();

                $transferDateCell = $detailsSheet->getCellByColumnAndRow(10, $row);
                $transferDate     = parse_excel_date_cell($transferDateCell);

                $beneficiaryNote  = trim((string)$detailsSheet->getCellByColumnAndRow(11, $row)->getValue());
                $licenseNumber    = trim((string)$detailsSheet->getCellByColumnAndRow(12, $row)->getValue());
                $purpose          = trim((string)$detailsSheet->getCellByColumnAndRow(13, $row)->getValue());

                $createdAtCell    = $detailsSheet->getCellByColumnAndRow(14, $row);
                $createdAtExcel   = parse_excel_date_cell($createdAtCell);

                // لو الصف فاضي تقريبًا نتخطاه
                if (
                    $beneficiaryName === '' &&
                    $ownerNumber === '' &&
                    $driverNumber === '' &&
                    $iban === '' &&
                    $tmsAmount == 0 &&
                    $bankAmount == 0 &&
                    $diffAmount == 0 &&
                    $transferDate === null
                ) {
                    continue;
                }

                $stmtDetail->bind_param(
                    "iissssssddsssss",
                    $uploaded_file_id,
                    $rowNumber,
                    $beneficiaryName,
                    $ownerNumber,
                    $driverNumber,
                    $ownerType,
                    $iban,
                    $tmsAmount,
                    $bankAmount,
                    $diffAmount,
                    $transferDate,
                    $beneficiaryNote,
                    $licenseNumber,
                    $purpose,
                    $createdAtExcel
                );
                $stmtDetail->execute();
                $insertedDetails++;
            }

            $stmtDetail->close();
        }
    }

    // تحديث حالة الملف
    $stmtUpd = $mysqli->prepare("UPDATE uploaded_files SET status='processed', processed_at=NOW() WHERE id=?");
    $stmtUpd->bind_param("i", $uploaded_file_id);
    $stmtUpd->execute();
    $stmtUpd->close();

    log_activity(
        'excel_processed',
        'تمت معالجة ملف التحويلات البنكية ID=' . $uploaded_file_id .
        ' (تفاصيل: ' . $insertedDetails . ')'
    );

    return true;
}

/**
 * محاولة ذكية لتحويل خلية تاريخ من Excel إلى نص yyyy-mm-dd
 */
function parse_excel_date_cell($cell): ?string
{
    $raw = $cell->getValue();

    if ($raw === null || $raw === '') {
        return null;
    }

    try {
        if (ExcelDate::isDateTime($cell)) {
            $dateObj = ExcelDate::excelToDateTimeObject($raw);
            return $dateObj->format('Y-m-d');
        }

        // محاولة تفسير القيمة كنص تاريخ عادي
        $ts = strtotime((string)$raw);
        if ($ts !== false) {
            return date('Y-m-d', $ts);
        }
    } catch (Throwable $e) {
        return null;
    }

    return null;
}
